<?php

namespace App\DataTables;

use App\Models\TrainingUser;
use App\Models\Training;
use App\Models\User;
use App\Models\InstructorInfo;
use Spatie\Permission\Models\Role;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;
use Illuminate\Http\Request;
use Auth;

class UserDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('checkbox', function($row){
                     return '<input type="checkbox" name="checkbox" class="tableChecked" value="'. $row->id .'" />';
                    })
        
            ->addColumn('action', function($row){
                            $url = url('admin/users/'.$row->id.'/edit');
                            if(Auth::user()->can('editUser')){
                            return '<a href="' . $url . '" class="btn btn-primary" ><i class="fas fa-edit"></i></a>';
                            }else {
                                return '';
                            }
            })
             ->editColumn('status', function($row){
                        return ($row->status == 1)?trans('home.yes'):trans('home.no');
             })
             ->editColumn('phone', function($row){
                        return ($row->type == 'student')? $row->student_info->mobile ?? '' : $row->phone ?? '';
             })
             ->filterColumn('f_name', function($query, $keyword) {
                        $sql = "f_name like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('email', function($query, $keyword) {
                        $sql = "email like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('phone', function($query, $keyword) {
                $query->where(function($subQuery) use ($keyword) {
                    $subQuery->where('phone', 'like', "%{$keyword}%")
                         ->orWhereHas('student_info', function($query) use ($keyword) {
                             $query->where('mobile', 'like', "%{$keyword}%");
                         });
                });
            })
             ->filterColumn('type', function($query, $keyword) {
                        $sql = "type like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('status', function($query, $keyword) {
                        $sql = "status like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->rawColumns(['action','checkbox'])
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(User $model): QueryBuilder
    {
        if($this->type && $this->type!="all"){
            return $model->select('users.*','students_info.mobile')->leftJoin('students_info','students_info.user_id','users.id')->where('type',$this->type)->newQuery();
        }
            return $model->select('users.*','students_info.mobile')->leftJoin('students_info','students_info.user_id','users.id')->newQuery();
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {   if(auth()->user()->can('table_export_buttons')) {
            if(auth()->user()->type == 'admin' || auth()->user()->type == 'super_admin') {
                $arr = ['excel','csv','print','pageLength'];
            }else if(auth()->user()->type == 'partner'){
                $arr = [];
            }else {
                $arr = ['print','pageLength'];
            }
        } else {
            $arr =[];
        }
        return $this->builder()
                    ->setTableId('users-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                     ->orderBy(1, 'asc') 
                    ->selectStyleSingle()
                    ->scrollX(true)
                    ->pageLength(50)
                    ->lengthMenu([['10','25','50','100','-1'],['10','25','50','100','All']])
                    ->parameters([
                        'dom'          => 'Bfrtip',
                        'buttons'      => $arr,
                        'columnDefs' => [
                            ['targets' => [-1], 'className' => 'text-right'],
                        ],
                        'language' => [
                                'url' => '//cdn.datatables.net/plug-ins/1.13.3/i18n/ar.json',
                        ],
                        'aoColumnDefs' => [
                            // Disable sorting and searching for "details" column
                            ['aTargets' => ['checkbox'], 'bSearchable' => false, 'bSortable' => false]
                        ]
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            [
                'title' => '',
                'data'           => 'checkbox',
                'name'           => 'checkbox',
                'class'          => 'msg_check',
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => false,
                'width'          => '10px',
            ],
            Column::make('id')
                    ->title(__('home.id')),
            Column::make('f_name')
                    ->title(__('home.name')),
            Column::make('email')
                    ->title(__('home.email')),
            Column::make('phone')
                    ->title(__('home.phone')),
            Column::make('type')
                    ->title(__('home.type')),
            Column::make('status')
                    ->title(__('home.status')),
            Column::make('action')
                    ->title(__('home.edit')),
            
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Users_' . date('YmdHis');
    }
}
