<?php

namespace App\DataTables;

use App\Models\TrainingCourse;
use App\Models\Training;
use App\Models\CenterHall;
use App\Models\Center;
use App\Models\Partner;
use App\Models\User;
use App\Models\Region;
use App\Models\InstructorInfo;
use App\Models\TrainingSurvey;
use App\Models\Instructor;
use App\Models\TrainingSchadual;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;
use Auth;
use DB;
use DateTime;

class TrainingDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        $object = (new EloquentDataTable($query))
            ->addColumn('checkbox', function($row){
                     return '<input type="checkbox" name="checkbox" class="tableChecked" value="'. $row->id .'" />';
                    })
        
            ->addColumn('action', function($row){
                            $now = new DateTime();
                            $start_date =  isset($row->trainings_schaduals->start_date) ? new DateTime($row->trainings_schaduals->start_date) : null ;
                            $end_date =  isset($row->trainings_schaduals->end_date) ? new DateTime($row->trainings_schaduals->end_date) : null ;
                            $url = url('admin/trainings/'.$row->id.'/edit');
                            if(Auth::user()->can('editTraining')){
                                if($start_date && ((($start_date) <= $now && $end_date >= $now) || $end_date < $now)){
                                    if(auth()->user()->type == 'super_admin') {
                                        return '<a href="' . $url . '" class="btn btn-primary" ><i class="fas fa-edit"></i></a>';
                                    } else {
                                        return '';
                                    }
                                } else {
                                    return '<a href="' . $url . '" class="btn btn-primary" ><i class="fas fa-edit"></i></a>';
                                }
                            }else {
                                return '';
                            }
             })
            ->addColumn('details', function($row){
                            $url = url('admin/trainings/'.$row->id);
                            if(Auth::user()->can('training_details')){
                                return '<a href="' . $url . '" class="btn btn-primary" ><i class="fas fa-info-circle"></i></a>';
                            }else {
                                return '';
                            }
                            
             })

             ->addColumn('n_students', function($row){
                        return count($row->students);
             })
             ->addColumn('instructor', function($row){
                 $instructor = TrainingCourse::where('training_id',$row->id)->first();
                 if($instructor) {
                     $user = User::find($instructor->instructor_id);
                     if($user){
                     return $user->name();
                     } else {
                         return '-';
                     }
                 } else if(isset($row->instructor_id)) {
                      $user = User::find($row->instructor_id);
                     if($user){
                     return $user->name();
                     } else {
                         return '-';
                     }
                 }
                 else {
                     return '-';
                 }
             })
             ->editColumn('created_by', function($row){
                     $user = User::find($row->created_by);
                     if($user){
                     return $user->name();
                     } else {
                         return '-';
                     }
             })
             ->addColumn('center', function($row){
                 $center = TrainingSchadual::where('training_id',$row->id)->first();
                 if($center){
                    $center = Center::find($center->center_id);
                    return app()->getLocale() == 'en' ? $center->name_en ?? '-' : $center->name_ar ?? '-' ;
                 } else {
                     return '';
                 }
             })
             ->addColumn('training_status', function($row){
                 if($row->training_status == '-1') {
                     return '<h4 class="d-flex justify-content-center mt-4 badge badge-pill badge-danger">'. trans('home.no_date_found') .'</h4>';
                 } else if($row->training_status == '0') {
                     return '<h2 class="d-flex justify-content-center mt-4 badge badge-pill badge-primary">'. trans('home.not_started') .'</h2>';
                 }
                 else if($row->training_status == '1') {
                     return '<h4 class="d-flex justify-content-center mt-4 badge badge-pill badge-success">'. trans('home.started') .'</h4>';
                 }
                 else if($row->training_status == '2') {
                     return '<h4 class="d-flex justify-content-center mt-4 badge badge-pill badge-warning">'. trans('home.finished') .'</h4>';
                 }
             })
             ->editColumn('instructor', function($row){
                        $user = User::find($row->instructor_id);
                        if($user) return $user->name(); else return '' ;
             })
             ->editColumn('region', function($row){
                 if(isset($row->trainings_schaduals) && $row->trainings_schaduals->hall_id != null){
                        $hall = CenterHall::find($row->trainings_schaduals->hall_id);
                        $region = Region::find($hall->region_id);
                        if($region){
                            return app()->getLocale() == 'en' ? $region->name_en ?? '-' : $region->name_ar ?? '-';
                        } else {
                            return '' ;
                        }
                } else {
                    return '' ;
                }
             })
             ->addColumn('link', function($row){
                        if($row->online_url) {
                         $res = $row->online_url ? '<a target="_blank" href="' . $row->online_url . '" class="btn btn-primary" ><i class="fas fa-link"></i></a>' : '';
                         } else {
                             $res = isset($row->trainings_schaduals->hall) ? '<a target="_blank" href="' . $row->trainings_schaduals->hall->map_url . '" class="btn btn-primary" ><i class="fas fa-map-marker-alt"></i></a>' : '';
                         }
                        return $res;
             });
             if(auth()->user()->can('training_instructor_survey')){
                 $object->addColumn('instructor_survey', function($row){
                            $training_survey = TrainingSurvey::where('training_id',$row->id)->whereHas('survey',function($q) {
                                $q->where('survay_type','instructor');
                            })->first();
                            if(isset($training_survey) && count($training_survey->survey->singleSurvayResult($training_survey->survey_id,auth()->user()->id,$row->id)) == 0) {
                                return '<a target="_blank" href="' . route('get.instructor.survey',$row->id) . '" class="btn btn-primary" ><i class="fas fa-poll"></i></a>';
                            } else {
                                return '';
                            }
                 });
             }
             if(auth()->user()->can('training_company_survey')){
                 $object->addColumn('company_survey', function($row){
                            $training_survey = TrainingSurvey::where('training_id',$row->id)->whereHas('survey',function($q) {
                                $q->where('survay_type','company');
                            })->first();
                            if(isset($training_survey)) {
                                return '<a target="_blank" href="' . route('get.company.survey',$row->id) . '" class="btn btn-primary" ><i class="fas fa-poll"></i></a>';
                            } else {
                                return '';
                            }
                 });
             }
             $object->addColumn('hall', function($row){
                        if($row->trainings_schaduals){
                            $hall = CenterHall::find($row->trainings_schaduals->hall_id);
                        }else{
                            $hall = null;
                        }
                        if($hall){
                            return app()->getLocale() == 'en' ? $hall->name_en ?? '-' : $hall->name_ar ?? '-';
                        } else {
                            return '' ;
                        }
             })
             ->editColumn('status', function($row){
                        return ($row->status == 1)?trans('home.yes'):trans('home.no');
             })
             ->addColumn('date_from', function($row){
                        if(!isset($row->trainings_schaduals->start_date)){
                            return '';
                        }
                        return \Carbon\Carbon::parse($row->trainings_schaduals->start_date)->format("Y-m-d");
             })
             ->editColumn('created_at', function($row){
                        if(!isset($row->created_at)){
                            return '';
                        }
                        return \Carbon\Carbon::parse($row->created_at)->format("Y-m-d");
             })
             ->addColumn('date_to', function($row){
                    if(!isset($row->trainings_schaduals->end_date)){
                        return '';
                    }
                   return \Carbon\Carbon::parse($row->trainings_schaduals->end_date)->format("Y-m-d");
             })
             ->orderColumn('instructor', function ($query, $order) {
                     $query->orderBy('trainings.instructor_id', $order);
                 })
             ->orderColumn('date_from', function ($query, $order) {
                     $query->where('start_date','<>','NULL')->orderBy('trainings_schaduals.start_date', $order);
                 })
             ->orderColumn('date_to', function ($query, $order) {
                     $query->where('end_date','<>','NULL')->orderBy('trainings_schaduals.end_date', $order);
                 })
             ->orderColumn('region', function ($query, $order) {
                     $query->orderBy('trainings.region_id', $order);
                 })
             ->orderColumn('link', function ($query, $order) {
                     $query->orderBy('trainings.online_url', $order)->orderBy('trainings_schaduals.hall_id', $order);
                 })
             ->orderColumn('n_students', function ($query, $order) {
                     $query->join('student_training_registered','student_training_registered.training_id','trainings.id')->orderBy('student_training_registered.training_id', $order);
                 })
             ->orderColumn('training_status', function ($query, $order) {
                     $query->where('start_date','<>','NULL')->orderBy('trainings_schaduals.start_date', $order)
                            ->where('end_date','<>','NULL')->orderBy('trainings_schaduals.end_date', $order);
                 })
             ->orderColumn('hall', function ($query, $order) {
                     $query->orderBy('hall_id', $order);
                 })
             ->orderColumn('name_ar', function ($query, $order) {
                     $query->orderBy('trainings.name_ar', $order);
                 })
            ->filterColumn('name_ar', function($query, $keyword) {
                    $sql = "trainings.name_ar like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('trainings.name_en', function($query, $keyword) {
                    $sql = "trainings.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('code', function($query, $keyword) {
                        $sql = "trainings.code like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('instructor', function($query, $keyword) {
                    $sql = "u1.f_name like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('created_by', function($query, $keyword) {
                    $sql = "u4.f_name like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('region', function($query, $keyword) {
                    $sql = "regions.name_ar like ? or regions.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%","%{$keyword}%"]);
                })
            ->filterColumn('hall', function($query, $keyword) {
                    $sql = "centers_halls.name_ar like ? or centers_halls.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%","%{$keyword}%"]);
                })
            ->filterColumn('center', function($query, $keyword) {
                    $sql = "centers.name_ar like ? or centers.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%","%{$keyword}%"]);
                })
            ->filterColumn('date_from',function($query, $keyword) {
                    $sql = "trainings_schaduals.start_date like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('date_to', function($query, $keyword) {
                    $sql = "trainings_schaduals.end_date like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('time_from', function($query, $keyword) {
                    $sql = "trainings.time_from like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('time_to', function($query, $keyword) {
                    $sql = "trainings.time_to like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('created_at', function($query, $keyword) {
                    $sql = "trainings.created_at like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('status', function($query, $keyword) {
                    if ($keyword == 'yes' || $keyword == '%yes%' || $keyword == 'نعم' || $keyword == '%نعم%') {
                        $query->where('trainings.status', 1);
                    } elseif ($keyword == 'no' || $keyword == '%no%' || $keyword == 'لا' || $keyword == '%لا%') {
                        $query->where('trainings.status', 0);
                    }
                })
            ->filterColumn('training_status', function($query, $keyword) {
                    $query->where(function ($query) use ($keyword) {
                        $words = explode(' ', $keyword);
                        foreach ($words as $word) {
                            $query->orWhere(function ($query) use ($word) {
                                if ($word == 'started' || str_contains('started',$word) || $word == 'سارية' || str_contains('سارية',$word)) {
                                    $query->whereHas('trainings_schaduals', function ($query) {
                                        $query->whereRaw('start_date <= now() and end_date >= now()');
                                    });
                                } elseif ($word == 'not started' || str_contains('not started',$word) || $word == 'لم تبدا بعد' || str_contains('لم تبدا بعد',$word)) {
                                    $query->whereHas('trainings_schaduals', function ($query) {
                                        $query->whereRaw('start_date > now()');
                                    });
                                    // dd($query->toSql());
                                } elseif ($word == 'finished' || str_contains('finished',$word) || $word == 'منتهية' || str_contains('منتهية',$word)) {
                                    $query->whereHas('trainings_schaduals', function ($query) {
                                        $query->whereRaw('end_date < now()');
                                    });
                                } elseif ($word == 'no date found' || str_contains('no date found',$word) || $word == 'لا يوجد تاريخ' || str_contains('لا يوجد تاريخ',$word)) {
                                    $query->
                                    doesntHave('trainings_schaduals','or')
                                    ->orWhereHas('trainings_schaduals', function ($query) {
                                        $query->whereNull('start_date')->orWhereNull('end_date');
                                    });
                                    // dd($query->toSql());
                                }
                            });
                        }
                    });
                })
            ->rawColumns(['link','company_survey','instructor_survey','date_from','date_to','training_status','action','details','checkbox'])
            ->setRowId('id');
            
            return $object;
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Training $model): QueryBuilder
    {
        if(auth()->user()->type=="partner"){
            $center_ids = Partner::where('user_id',auth()->user()->id)->pluck('center_id')->toArray();
            // $center_ids = auth()->user()->partner->centers->pluck('id')->toArray();
            $training_ids = TrainingSchadual::whereIn('center_id',$center_ids)->pluck('training_id')->toArray();
            return $model
            ->leftJoin('regions as regions','regions.id','trainings.region_id')
            ->leftJoin('trainings_schaduals as trainings_schaduals','trainings_schaduals.training_id','trainings.id')
            ->leftJoin('centers as centers','centers.id','trainings_schaduals.center_id')
            ->leftJoin('centers_halls as centers_halls','centers_halls.id','trainings_schaduals.hall_id')
            ->leftJoin('users as u1','u1.id','trainings.instructor_id')
            ->leftJoin('users as u2','u2.id','trainings.coordinator_id')
            ->leftJoin('users as u3','u3.id','trainings.marketer_id')
            ->leftJoin('users as u4','u4.id','trainings.created_by')
            ->select('trainings.*','u1.f_name as instructor_name','u2.f_name as coordinator_name','u3.f_name as marketer_name','u4.f_name as creator','regions.name_ar as region_name_ar','regions.name_en as region_name_en','centers_halls.name_ar as center_hall_name_ar','centers_halls.name_en as center_hall_name_en','centers.name_ar as center_name_ar','centers.name_en as center_name_en')
            ->whereIn('trainings.id',$training_ids)
            ->where('trainings.status',1)
            ->newQuery();
        }
        if(auth()->user()->type=="instructor"){
            // $instructor_id = InstructorInfo::where('user_id',auth()->user()->id)->first()->id;
            $training_ids = TrainingCourse::where('instructor_id',auth()->user()->id)->pluck('training_id')->toArray();
            return $model
            ->leftJoin('regions as regions','regions.id','trainings.region_id')
            ->leftJoin('trainings_schaduals as trainings_schaduals','trainings_schaduals.training_id','trainings.id')
            ->leftJoin('centers as centers','centers.id','trainings_schaduals.center_id')
            ->leftJoin('centers_halls as centers_halls','centers_halls.id','trainings_schaduals.hall_id')
            ->leftJoin('users as u1','u1.id','trainings.instructor_id')
            ->leftJoin('users as u2','u2.id','trainings.coordinator_id')
            ->leftJoin('users as u3','u3.id','trainings.marketer_id')
            ->leftJoin('users as u4','u4.id','trainings.created_by')
            ->select('trainings.*','u1.f_name as instructor_name','u2.f_name as coordinator_name','u3.f_name as marketer_name','u4.f_name as creator','regions.name_ar as region_name_ar','regions.name_en as region_name_en','centers_halls.name_ar as center_hall_name_ar','centers_halls.name_en as center_hall_name_en','centers.name_ar as center_name_ar','centers.name_en as center_name_en')
            ->whereIn('trainings.id',$training_ids)
            ->where('trainings.status',1)
            ->newQuery();
        }
        if(auth()->user()->type=="coordinator"){
            return $model
            ->leftJoin('regions as regions','regions.id','trainings.region_id')
            ->leftJoin('trainings_schaduals as trainings_schaduals','trainings_schaduals.training_id','trainings.id')
            ->leftJoin('centers_halls as centers_halls','centers_halls.id','trainings_schaduals.hall_id')
            ->leftJoin('centers as centers','centers.id','trainings_schaduals.center_id')
            ->leftJoin('users as u1','u1.id','trainings.instructor_id')
            ->leftJoin('users as u2','u2.id','trainings.coordinator_id')
            ->leftJoin('users as u3','u3.id','trainings.marketer_id')
            ->leftJoin('users as u4','u4.id','trainings.created_by')
            ->select('trainings.*','u1.f_name as instructor_name','u2.f_name as coordinator_name','u3.f_name as marketer_name','u4.f_name as creator','regions.name_ar as region_name_ar','regions.name_en as region_name_en','centers_halls.name_ar as center_hall_name_ar','centers_halls.name_en as center_hall_name_en','centers.name_ar as center_name_ar','centers.name_en as center_name_en')
            ->where('coordinator_id',auth()->user()->id)
            ->where('trainings.status',1)
            ->newQuery();
        }
        if(auth()->user()->type=="marketer"){
            return $model
            ->leftJoin('regions as regions','regions.id','trainings.region_id')
            ->leftJoin('trainings_schaduals as trainings_schaduals','trainings_schaduals.training_id','trainings.id')
            ->leftJoin('centers as centers','centers.id','trainings_schaduals.center_id')
            ->leftJoin('centers_halls as centers_halls','centers_halls.id','trainings_schaduals.hall_id')
            ->leftJoin('users as u1','u1.id','trainings.instructor_id')
            ->leftJoin('users as u2','u2.id','trainings.coordinator_id')
            ->leftJoin('users as u3','u3.id','trainings.marketer_id')
            ->leftJoin('users as u4','u4.id','trainings.created_by')
            ->select('trainings.*','u1.f_name as instructor_name','u2.f_name as coordinator_name','u3.f_name as marketer_name','u4.f_name as creator','regions.name_ar as region_name_ar','regions.name_en as region_name_en','centers_halls.name_ar as center_hall_name_ar','centers_halls.name_en as center_hall_name_en','centers.name_ar as center_name_ar','centers.name_en as center_name_en')
            ->where('marketer_id',auth()->user()->id)->where('trainings.status',1)
            ->newQuery();
        }
        else{
            return $model
            ->leftJoin('regions as regions','regions.id','trainings.region_id')
            ->leftJoin('trainings_schaduals as trainings_schaduals','trainings_schaduals.training_id','trainings.id')
            ->leftJoin('centers as centers','centers.id','trainings_schaduals.center_id')
            ->leftJoin('centers_halls as centers_halls','centers_halls.id','trainings_schaduals.hall_id')
            ->leftJoin('users as u1','u1.id','trainings.instructor_id')
            ->leftJoin('users as u2','u2.id','trainings.coordinator_id')
            ->leftJoin('users as u3','u3.id','trainings.marketer_id')
            ->leftJoin('users as u4','u4.id','trainings.created_by')
            // ->leftJoin('student_training_registered as std_tr_reg','std_tr_reg.training_id','trainings.id')
            ->select('trainings.*','u1.f_name as instructor_name','u2.f_name as coordinator_name','u3.f_name as marketer_name','u4.f_name as creator','regions.name_ar as region_name_ar','regions.name_en as region_name_en','centers_halls.name_ar as center_hall_name_ar','centers_halls.name_en as center_hall_name_en','centers.name_ar as center_name_ar','centers.name_en as center_name_en')
            ->newQuery();
        }
        
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {   if(auth()->user()->can('table_export_buttons')) {
            if(auth()->user()->type == 'admin' || auth()->user()->type == 'super_admin') {
                $arr = ['excel','csv','print','pageLength'];
            }else if(auth()->user()->type == 'partner'){
                $arr = [];
            }else {
                $arr = ['print','pageLength'];
            }
        } else {
            $arr =[];
        }
        return $this->builder()
                    ->setTableId('trainings-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                     ->orderBy(1, 'asc') 
                    ->selectStyleSingle()
                    ->scrollX(true)
                    ->pageLength(50)
                    ->lengthMenu([['10','25','50','100','-1'],['10','25','50','100','All']])
                    ->parameters([
                        'dom'          => 'Bfrtip',
                        'buttons'      => $arr,
                        'columnDefs' => [
                            ['targets' => [-1], 'className' => 'text-right'],
                        ],
                        'language' => [
                                'url' => '//cdn.datatables.net/plug-ins/1.13.3/i18n/ar.json',
                        ],
                        'aoColumnDefs' => [
                            // Disable sorting and searching for "details" column
                            ['aTargets' => ['checkbox'], 'bSearchable' => false, 'bSortable' => false]
                        ]
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        $columns = [
            [
                'title' => '',
                'data'           => 'checkbox',
                'name'           => 'checkbox',
                'class'          => 'msg_check',
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => false,
                'width'          => '10px',
            ],
            Column::make('code')
                    ->title(__('home.code')),
            Column::make('name_ar')
                    ->title(__('home.name_ar')),
            Column::make('center')
                    ->title(__('home.partner')),
            Column::make('instructor')
                    ->title(__('home.instructor')),];
            if(auth()->user()->can('training_instructor_survey')){
                $instructor_survey = array(Column::make('instructor_survey')->title(__('home.instructor_survey')));
                $columns = array_merge($columns,$instructor_survey);
            }
            if(auth()->user()->can('training_company_survey')){
                $company_survey = array(Column::make('company_survey')->title(__('home.company_survey')));
                $columns = array_merge($columns,$company_survey);
            }
            array_push($columns,
                    Column::make('region')
                            ->title(__('home.region')),
                    Column::make('hall')
                            ->title(__('home.hall')),
                    Column::make('link')
                            ->title(__('home.online_link_or_location')),
                    // Column::make('overall_days')
                    //         ->title(__('home.overall_days')),
                    // Column::make('overall_time')
                    //         ->title(__('home.overall_time')),
                    Column::make('timing_type')
                            ->title(__('home.timing_type')),
                    Column::make('date_from')
                            ->title(__('home.date_from')),
                    Column::make('date_to')
                            ->title(__('home.date_to')),
                    Column::make('time_from')
                            ->title(__('home.time_from')),
                    Column::make('time_to')
                            ->title(__('home.time_to')),
                    Column::make('n_students')
                            ->title(__('home.n_students')),
                    Column::make('created_at')
                            ->title(__('home.created_at')),
                    Column::make('created_by')
                            ->title(__('home.created_by')),
                    Column::make('training_status')
                            ->title(__('home.training_status')),
                    Column::make('status')
                            ->title(__('home.status'))
            );
            
            array_push($columns,
                    Column::make('details')
                            ->title(__('home.training_details')),
                    Column::make('action')
                            ->title(__('home.edit'))
            );
        
        return $columns;
            
        }

            // if(auth()->user()->roles->pluck('name')->toArray()[0] == 'منسق خارجي') {
    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Trainings_' . date('YmdHis');
    }
}
