<?php

namespace App\DataTables;

use App\Models\TrainingCourse;
use App\Models\User;
use App\Models\Lesson;
use App\Models\InstructorInfo;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;
use Auth;

class LessonDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('checkbox', function($row){
                     return '<input type="checkbox" name="checkbox" class="tableChecked" value="'. $row->id .'" />';
                    })
        
            ->addColumn('action', function($row){
                            $url = url('admin/lessons/'.$row->id.'/edit');
                            if(Auth::user()->can('editLesson')){
                            return '<a href="' . $url . '" class="btn btn-primary" ><i class="fas fa-edit"></i></a>';
                            }else {
                                return '';
                            }
                    })
            ->addColumn('attachments', function($row){
                
                if($row->attachments) {
                    $feed = '';
                    if($row->attachments->{app()->getLocale() == 'en' ? 'intro_attachment_en' : 'intro_attachment_ar'} ) {
                    $feed .= __('home.intro_attachment')
                    .'<a target="_blank" href="' . url('uploads/courses/attachments/files/'.$row->attachments->{app()->getLocale() == 'en' ? 'intro_attachment_en' : 'intro_attachment_ar'} )
                    .'">
                          <i class="fas fa-eye"></i>
                        '.$row->attachments->{app()->getLocale() == 'en' ? 'intro_attachment_en' : 'intro_attachment_ar'} . ' </a> <br>';
                    }
                    if($row->attachments->{ app()->getLocale() == 'en' ? 'attachment_en' : 'attachment_ar' } ) { 
                    $feed .= __('home.attachment')
                    .'<a target="_blank" href="' . url('uploads/courses/attachments/files/'.$row->attachments->{app()->getLocale() == 'en' ? 'attachment_en' : 'attachment_ar' } ) . '">
                          <i class="fas fa-eye"></i>
                        '.$row->attachments->{app()->getLocale() == 'en' ? 'attachment_en' : 'attachment_ar'} .'</a> <br>';
                    }
                    
                    if($row->attachments->{app()->getLocale() == 'en' ? 'details_attachment_en' : 'details_attachment_ar'} ) {
                    $feed .= __('home.details_attachment')
                    . '<a target="_blank" href="'. url('uploads/courses/attachments/files/'.$row->attachments->{ app()->getLocale() == 'en' ? 'details_attachment_en' : 'details_attachment_ar' } ).'">
                          <i class="fas fa-eye"></i>
                        '. $row->attachments->{app()->getLocale() == 'en' ? 'details_attachment_en' : 'details_attachment_ar'} .'</a>';
                    }
                    return $feed;
                }
            })
            ->addColumn('text', function($row){
                         return strip_tags(mb_strimwidth($row->{app()->getLocale() == 'en' ? 'text_en' : 'text_ar'}, 0, 100, "..."));
            })
            ->addColumn('image', function($row){
                        if($row->image){
                            $url = url('uploads/courses/lesson_images/'.$row->image);
                            return '<img src="' . $url . '" border="0" width="40" class="img-rounded" />';
                        }else{
                            $url = url("resources/assets/back/img/noimage.png");
                            return '<img src="' . $url . '" border="0" width="40" class="img-rounded" />';
                        }
                    })
            ->addColumn('course', function($row){
                        return $row->course->{app()->getLocale() == 'en' ? 'name_en' : 'name_ar'};
            })
            ->editColumn('status', function($row){
                        return ($row->status == 1)?trans('home.yes'):trans('home.no');
            })
            ->filterColumn('name_ar', function($query, $keyword) {
                    $sql = "courses_lessons.name_ar like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('name_en', function($query, $keyword) {
                    $sql = "courses_lessons.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('text', function($query, $keyword) {
                    $sql = "courses_lessons.text_ar like ? or courses_lessons.text_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%","%{$keyword}%"]);
                })
            ->filterColumn('course', function($query, $keyword) {
                    $sql = "courses.name_ar like ? or courses.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%","%{$keyword}%"]);
                })
            ->filterColumn('status', function($query, $keyword) {
                        if ($keyword == 'yes' || $keyword == '%yes%' || $keyword == 'نعم' || $keyword == '%نعم%') {
                            $query->where('courses_lessons.status', 1);
                        } elseif ($keyword == 'no' || $keyword == '%no%' || $keyword == 'لا' || $keyword == '%لا%') {
                            $query->where('courses_lessons.status', 0);
                        }
            })
            ->rawColumns(['attachments','image','action','checkbox'])
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Lesson $model): QueryBuilder
    {
        if(auth()->user()->type=="instructor"){
            // $instructor_id = InstructorInfo::where('user_id',auth()->user()->id)->first()->id;
            $course_ids = TrainingCourse::where('instructor_id',auth()->user()->id)->pluck('course_id')->toArray();
            return $model->leftJoin('courses','courses.id','courses_lessons.course_id')
            ->select('courses_lessons.*','courses.name_ar as course_name_ar','courses.name_en as course_name_en')
            ->whereIn('lessons.course_id',$course_ids)->newQuery();
        }else {
            return $model->leftJoin('courses','courses.id','courses_lessons.course_id')
            ->select('courses_lessons.*','courses.name_ar as course_name_ar','courses.name_en as course_name_en')
            ->newQuery();
        }
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {   if(auth()->user()->can('table_export_buttons')) {
            if(auth()->user()->type == 'admin' || auth()->user()->type == 'super_admin') {
                $arr = ['excel','csv','print','pageLength'];
            }else if(auth()->user()->type == 'partner'){
                $arr = [];
            }else {
                $arr = ['print','pageLength'];
            }
        } else {
            $arr =[];
        }
        return $this->builder()
                    ->setTableId('lessons-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                     ->orderBy(1, 'asc') 
                    ->selectStyleSingle()
                    ->scrollX(true)
                    ->pageLength(50)
                    ->lengthMenu([['10','25','50','100','-1'],['10','25','50','100','All']])
                    ->parameters([
                        'dom'          => 'Bfrtip',
                        'buttons'      => $arr,
                        'columnDefs' => [
                            ['targets' => [-1], 'className' => 'text-right'],
                        ],
                        'language' => [
                                'url' => '//cdn.datatables.net/plug-ins/1.13.3/i18n/ar.json',
                        ],
                        'aoColumnDefs' => [
                            // Disable sorting and searching for "details" column
                            ['aTargets' => ['checkbox'], 'bSearchable' => false, 'bSortable' => false]
                        ]
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            [
                'title' => '',
                'data'           => 'checkbox',
                'name'           => 'checkbox',
                'class'          => 'msg_check',
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => false,
                'width'          => '10px',
            ],
            Column::make('id')
                    ->title(__('home.id')),
            Column::make('name_en')
                    ->title(__('home.name_en')),
            Column::make('name_ar')
                    ->title(__('home.name_ar')),
            Column::make('course')
                    ->title(__('home.course')),
            Column::make('image')
                    ->title(__('home.image')),
            Column::make('attachments')
                    ->title(__('home.attachments')),
            Column::make('text')
                    ->title(__('home.text')),
            Column::make('status')
                    ->title(__('home.status')),
            Column::make('action')
                    ->title(__('home.edit')),
            
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Lessons_' . date('YmdHis');
    }
}
