<?php

namespace App\DataTables;

use App\Models\Partner;
use App\Models\TrainingSchadual;
use App\Models\TrainingCourse;
use App\Models\Training;
use App\Models\User;
use App\Models\InstructorInfo;
use App\Models\Course;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;
use Auth;

class CourseDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('checkbox', function($row){
                     return '<input type="checkbox" name="checkbox" class="tableChecked" value="'. $row->id .'" />';
                    })
        
            ->addColumn('action', function($row){
                            $url = url('admin/courses/'.$row->id.'/edit');
                            if(Auth::user()->can('editCourse')){
                            return '<a href="' . $url . '" class="btn btn-primary" ><i class="fas fa-edit"></i></a>';
                            }else {
                                return '';
                            }
                    })
            ->addColumn('image', function($row){
                        if($row->image){
                            $url = url('uploads/courses/images/'.$row->image);
                            return '<img src="' . $url . '" border="0" width="40" class="img-rounded" />';
                        }else{
                            $url = url("resources/assets/back/img/noimage.png");
                            return '<img src="' . $url . '" border="0" width="40" class="img-rounded" />';
                        }
                    })
            ->addColumn('lessons', function($row){
                        return '<a href="'. route("lessons.show", $row->id) .'">
                                    <svg xmlns="http://www.w3.org/2000/svg" height="12" width="13.5" viewBox="0 0 576 512">
                                        <path fill="#4A91CC" d="M288 32c-80.8 0-145.5 36.8-192.6 80.6C48.6 156 17.3 208 2.5 243.7c-3.3 7.9-3.3 16.7 0 24.6C17.3 304 48.6 356 95.4 399.4C142.5 443.2 207.2 480 288 480s145.5-36.8 192.6-80.6c46.8-43.5 78.1-95.4 93-131.1c3.3-7.9 3.3-16.7 0-24.6c-14.9-35.7-46.2-87.7-93-131.1C433.5 68.8 368.8 32 288 32zM144 256a144 144 0 1 1 288 0 144 144 0 1 1 -288 0zm144-64c0 35.3-28.7 64-64 64c-7.1 0-13.9-1.2-20.3-3.3c-5.5-1.8-11.9 1.6-11.7 7.4c.3 6.9 1.3 13.8 3.2 20.7c13.7 51.2 66.4 81.6 117.6 67.9s81.6-66.4 67.9-117.6c-11.1-41.5-47.8-69.4-88.6-71.1c-5.8-.2-9.2 6.1-7.4 11.7c2.1 6.4 3.3 13.2 3.3 20.3z"/></svg>
                                    '. __("home.see_lessons").' (' . count($row->lessons ?? 0) .')
                                </a>';
                    })
            ->addColumn('student_training_bag', function($row){
                        if($row->student_training_bag)
                            return '<a target="_blank" href="' . url('uploads/courses/training_bags/files/'.$row->student_training_bag ) . '">
                                          <i class="fas fa-eye"></i>
                                        '.$row->student_training_bag .'</a> <br>';
                                        
                        else return '';
                    })
            ->addColumn('instructor_training_bag', function($row){
                        if($row->instructor_training_bag)
                            return '<a target="_blank" href="' . url('uploads/courses/training_bags/files/'.$row->instructor_training_bag ) . '">
                                          <i class="fas fa-eye"></i>
                                        '.$row->instructor_training_bag .'</a> <br>';
                                        
                        else return '';
                    })
        
             ->editColumn('status', function($row){
                        return ($row->status == 1)?trans('home.yes'):trans('home.no');
             })
             ->filterColumn('name_ar', function($query, $keyword) {
                    $sql = "name_ar like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('code', function($query, $keyword) {
                    $sql = "code like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('status', function($query, $keyword) {
                        if ($keyword == 'yes' || $keyword == '%yes%' || $keyword == 'نعم' || $keyword == '%نعم%') {
                            $query->where('status', 1);
                        } elseif ($keyword == 'no' || $keyword == '%no%' || $keyword == 'لا' || $keyword == '%لا%') {
                            $query->where('status', 0);
                        }
                })
            ->rawColumns(['instructor_training_bag','student_training_bag','lessons','image','action','checkbox'])
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Course $model): QueryBuilder
    {
        if(auth()->user()->type=="partner"){
            $center_ids = Partner::where('user_id',auth()->user()->id)->pluck('center_id')->toArray();
            $training_ids = TrainingSchadual::whereIn('center_id',$center_ids)->pluck('training_id')->toArray();
            $course_ids = TrainingCourse::whereIn('training_id',$training_ids)->pluck('course_id')->toArray();
            return $model->whereIn('courses.id',$course_ids)->newQuery();
        }
        if(auth()->user()->type=="instructor"){
            // $instructor_id = InstructorInfo::where('user_id',auth()->user()->id)->first()->id;
            $course_ids = TrainingCourse::where('instructor_id',auth()->user()->id)->pluck('course_id')->toArray();
            return $model->whereIn('courses.idid',$course_ids)->newQuery();
        }
        if(auth()->user()->type=="coordinator"){
            $training_ids = Training::where('coordinator_id',auth()->user()->id)->pluck('id')->toArray();
            $course_ids = TrainingCourse::whereIn('training_id',$training_ids)->pluck('course_id')->toArray();
            return $model->whereIn('courses.idid',$course_ids)->newQuery();
        }
        if(auth()->user()->type=="marketer"){
            $training_ids = Training::where('marketer_id',auth()->user()->id)->pluck('id')->toArray();
            $course_ids = TrainingCourse::whereIn('training_id',$training_ids)->pluck('course_id')->toArray();
            return $model->whereIn('courses.idid',$course_ids)->newQuery();
        }else {
            return $model->newQuery();
        }
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {   if(auth()->user()->can('table_export_buttons')) {
            if(auth()->user()->type == 'admin' || auth()->user()->type == 'super_admin') {
                $arr = ['excel','csv','print','pageLength'];
            }else if(auth()->user()->type == 'partner'){
                $arr = [];
            }else {
                $arr = ['print','pageLength'];
            }
        } else {
            $arr =[];
        }
        return $this->builder()
                    ->setTableId('courses-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                     ->orderBy(1, 'asc') 
                    ->selectStyleSingle()
                    ->scrollX(true)
                    ->pageLength(50)
                    ->lengthMenu([['10','25','50','100','-1'],['10','25','50','100','All']])
                    ->parameters([
                        'dom'          => 'Bfrtip',
                        'buttons'      => $arr,
                        'columnDefs' => [
                            ['targets' => [-1], 'className' => 'text-right'],
                        ],
                        'language' => [
                                'url' => '//cdn.datatables.net/plug-ins/1.13.3/i18n/ar.json',
                        ],
                        'aoColumnDefs' => [
                            // Disable sorting and searching for "details" column
                            ['aTargets' => ['checkbox'], 'bSearchable' => false, 'bSortable' => false]
                        ]
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            [
                'title' => '',
                'data'           => 'checkbox',
                'name'           => 'checkbox',
                'class'          => 'msg_check',
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => false,
                'width'          => '10px',
            ],
            Column::make('code')
                    ->title(__('home.course_code')),
            // Column::make('name_en')
            //         ->title(__('home.name_en')),
            Column::make('name_ar')
                    ->title(__('home.name_ar')),
            // Column::make('image')
            //         ->title(__('home.image')),
            // Column::make('lessons')
            //         ->title(__('home.lessons')),
            Column::make('student_training_bag')
                    ->title(__('home.student_training_bag')),
            Column::make('instructor_training_bag')
                    ->title(__('home.instructor_training_bag')),
            Column::make('status')
                    ->title(__('home.status')),
            Column::make('action')
                    ->title(__('home.edit/add_bags')),
            
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Courses_' . date('YmdHis');
    }
}
